/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef MX__THREAD_H
#define MX__THREAD_H

#ifndef MX_AUTO_CONFIG_H
#include "mx_auto_config.h"
#endif

#ifdef MX_KERNEL

#include "mx_klib.h"

#define MX__MUTEX_T mx_klib_mutex_t
#define MX__MUTEX_INIT(m) mx_klib_mutex_init(m)
#define MX__MUTEX_DESTROY(m) mx_klib_mutex_destroy(m)
#define MX__MUTEX_LOCK(m) mx_klib_mutex_lock(m)
#define MX__MUTEX_UNLOCK(m) mx_klib_mutex_unlock(m)

#define MX__THREAD_T mx_klib_thread_t
#define MX__THREAD_RETURN_T mx_klib_thread_return_t
#define MX__THREAD_CREATE(threadp, fctn, arg) \
mx_klib_thread_create(threadp, fctn, arg)
#define MX__THREAD_JOIN(threadp) mx_klib_thread_join(threadp)
#define MX__THREAD_EXIT_WITH_NOTHING(threadp) mx_klib_thread_exit(threadp)
/*#define MX__THREAD_SELF()  useless now */
#define MX__THREAD_INIT(threadp) mx_klib_thread_init(threadp, "mx_klib")

#define MX__EVENT_T mx_sync_t
#define MX__EVENT_INIT(eventp) mx_sync_init(eventp, NULL, -1, "mx_klib_event")
#define MX__EVENT_DESTROY(eventp) mx_sync_destroy(eventp)
#define MX__EVENT_SIGNAL(eventp) mx_wake(eventp)

#define MX__EVENT_WAIT(eventp, lock) do {	\
  mx_klib_mutex_unlock(lock);			\
  mx_sleep(eventp, MX_MAX_WAIT, MX_SLEEP_INTR);	\
  mx_klib_mutex_lock(lock);			\
} while(0)                                                                      

#define MX__EVENT_WAIT_TIMEOUT(eventp, lock, msecs)	\
do {							\
  mx_klib_mutex_unlock(lock);				\
  mx_sleep(eventp, msecs, MX_SLEEP_INTR);		\
  mx_klib_mutex_lock(lock);				\
} while(0)                                                                      

/* semaphore useless now */

#else /* MX_KERNEL */

#include "mx_auto_config.h"
#include "myriexpress.h"

#if MX_THREAD_SAFE

#if (MX_OS_LINUX || MX_OS_FREEBSD || MX_OS_MACOSX || MX_OS_SOLARIS || MX_OS_UDRV)

#include <pthread.h>
#include <sys/time.h>

#define MX__MUTEX_T pthread_mutex_t
#define MX__MUTEX_INIT(lock) pthread_mutex_init(lock,NULL)
#define MX__MUTEX_DESTROY(lock) pthread_mutex_destroy(lock)
#define MX__MUTEX_LOCK(lock) pthread_mutex_lock(lock)
#define MX__MUTEX_UNLOCK(lock) pthread_mutex_unlock(lock)

#define MX__THREAD_T pthread_t
#define MX__THREAD_RETURN_T void*
#define MX__THREAD_CREATE(threadp,fctn,arg) \
pthread_create(threadp, NULL, fctn, arg)
#define MX__THREAD_JOIN(threadp) pthread_join(*threadp, NULL)
#define MX__THREAD_EXIT_WITH_NOTHING(threadp) pthread_exit(0); return 0
#define MX__THREAD_SELF() pthread_self()
#define MX__THREAD_CANCEL(threadp) pthread_cancel(*threadp)
#define MX__THREAD_INIT(threadp) do { /* nothing */ } while (0)

#if MX_OS_SOLARIS
#include <semaphore.h>
#endif

#define MX__SEMAPHORE_T sem_t
#define MX__SEMAPHORE_WAIT(sema) sem_wait(sema)
#define MX__SEMAPHORE_INIT(sema,value) sem_init(sema,0,value)
#define MX__SEMAPHORE_DESTROY(sema) sem_destroy(sema)
#define MX__SEMAPHORE_POST(sema) sem_post(sema)

#define MX__EVENT_T pthread_cond_t
#define MX__EVENT_INIT(eventp) pthread_cond_init (eventp, NULL)
#define MX__EVENT_SIGNAL(eventp) pthread_cond_signal (eventp)
#define MX__EVENT_WAIT(eventp, lock) pthread_cond_wait (eventp, lock)
#define MX__EVENT_WAIT_TIMEOUT(eventp, lock, milli)		\
do {								\
  struct timeval tv;						\
  struct timespec abstime;					\
  gettimeofday(&tv, NULL);					\
  abstime.tv_sec = tv.tv_sec + (milli/1000);			\
  abstime.tv_nsec = tv.tv_usec * 1000 + (milli%1000) * 1000000;	\
  pthread_cond_timedwait(eventp, lock, &abstime);		\
} while (0)
#define MX__EVENT_DESTROY(eventp) pthread_cond_destroy (eventp)

#elif MX_OS_WINNT

#include <windows.h>
#include <process.h>

#define MX__MUTEX_T HANDLE
#define MX__MUTEX_INIT(lock) *lock=CreateMutex(NULL,FALSE,NULL)
#define MX__MUTEX_DESTROY(lock) CloseHandle(*lock)
#define MX__MUTEX_LOCK(lock) WaitForSingleObject(*lock, INFINITE)
#define MX__MUTEX_UNLOCK(lock) ReleaseMutex(*lock)

typedef struct mx__threadinfo_t
{
  void *(*start_routine)(void*);
  void *arg;
} mx__threadinfo_t;

typedef struct mx__thread_t
{
  HANDLE thread;
  mx__threadinfo_t *ti;
} mx__thread_t;

int mx__thread_create(mx__thread_t *thread,
		      void *(*start_routine)(void *), void *arg);
int mx__thread_join(mx__thread_t *t, void **thread_return);
void mx__thread_exit(void *retval);

#define MX__THREAD_T mx__thread_t
#define MX__THREAD_RETURN_T void*
#define MX__THREAD_CREATE(thread,fctn,arg)                             \
mx__thread_create(thread, fctn, arg)
#define MX__THREAD_JOIN(thread) mx__thread_join(thread, NULL)
#define MX__THREAD_EXIT_WITH_NOTHING(retval) mx__thread_exit(retval); return 0
#define MX__THREAD_SELF() mx__thread_self()
#define MX__THREAD_INIT(threadp) do { /* nothing */ } while (0)

#define MX__SEMAPHORE_T HANDLE
#define MX__SEMAPHORE_WAIT(sema) WaitForSingleObject(*sema, INFINITE)
#define MX__SEMAPHORE_INIT(sema,value)                                 \
*sema = CreateSemaphore(NULL, value, 0x7FFFFFFFL, NULL)
#define MX__SEMAPHORE_DESTROY(sema) CloseHandle(*sema)
#define MX__SEMAPHORE_POST(sema) ReleaseSemaphore(*sema, 1, NULL)

#define MX__EVENT_T HANDLE
#define MX__EVENT_INIT(event) *event=CreateEvent(NULL,FALSE,FALSE,NULL)
#define MX__EVENT_SIGNAL(event) SetEvent(*event)
#define MX__EVENT_WAIT(event,lock) do {			\
   SignalObjectAndWait(*(lock),*(event), INFINITE,FALSE);	\
   WaitForSingleObject(*(lock), INFINITE);		\
  } while (0);
#define MX__EVENT_WAIT_TIMEOUT(event,lock,milli) do {	\
   SignalObjectAndWait(*(lock), *(event), milli, FALSE);	\
   WaitForSingleObject(*(lock), INFINITE);		\
  } while (0);
#define MX__EVENT_DESTROY(event) CloseHandle(*event)

#else
#error no thread support for this OS
#endif

extern MX__MUTEX_T Mx_lock;
#define MX_LOCK &Mx_lock

#else

#define MX__MUTEX_T int
#define MX__MUTEX_INIT(x)
#define MX__MUTEX_DESTROY(x)
#define MX__MUTEX_LOCK(x)
#define MX__MUTEX_UNLOCK(x)

#define MX__THREAD_T int
#define MX__THREAD_RETURN_T int
#define MX__THREAD_CREATE(threadp,fctn,arg)
#define MX__THREAD_JOIN(threadp)
#define MX__THREAD_EXIT_WITH_NOTHING(threadp) return 0
#define MX__THREAD_SELF()
#define MX__THREAD_INIT(threadp)

#define MX__SEMAPHORE_T int
#define MX__SEMAPHORE_WAIT(sema)
#define MX__SEMAPHORE_INIT(sema,value)
#define MX__SEMAPHORE_DESTROY(sema)
#define MX__SEMAPHORE_POST(sema)

#define MX__EVENT_T int
#define MX__EVENT_INIT(eventp)
#define MX__EVENT_SIGNAL(eventp)
#define MX__EVENT_WAIT(eventp,lock)
#define MX__EVENT_WAIT_TIMEOUT(eventp,lock,milli)
#define MX__EVENT_DESTROY(eventp)

#define MX_LOCK

#endif

#endif /* MX_KERNEL */

#endif /* MX_THREAD_H */
